#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdbool.h>
#include <syslog.h>
#include <stdlib.h>
#include <getopt.h>
#include "common.h"
#include "ctl.h"
#include "wds.h"
#include "lite-qmi-wds.h"
#include <QmuxTransport.h>
#include <QmiService.h>
#include <CtlService.h>
#include <QmiSyncObject.h>

#include <locale.h>
#include <sys/time.h>
#include "msgid.h"

#include "dev_util.h"
#include "qmux_util.h"
#include "str_util.h"

extern void wds_indication_handler(uint16_t msgid, uint8_t *msg, uint32_t rlen);
extern void SetKeepAliveDataSession(QmiService* pQmiService, const bool enableKA);
extern void ResetAndModifyProfileSettings(QmiService* pService, uint8_t profileID, uint8_t profileType, wds_profileInfo *pProfile);
extern void GetApnOpReservedPcoList(QmiService* pQmiService, char *pApnName);
extern void GetApnMsisdnInfo(QmiService* pQmiService, char *pApnName);
extern void DeleteAllProfiles(QmiService* pQmiService, uint64_t ProfileTypeMask, uint64_t *pProfilePersistenceMask, uint64_t *pProfileClientTypeMask);
extern void SetEhrpdFallbackApnList(QmiService* pQmiService, uint8_t fallback_apn_name_list_len, WdsEhrpdFallbackApn_t *pWdsEhrpdFallbackApn);
extern void SetModemAssistedKaStart(QmiService* pQmiService, pack_wds_modem_assisted_ka_start_t *pReqParam);
extern void SetModemAssistedKaStop(QmiService* pQmiService, uint32_t  KeepAliveHandle);
extern void GetEhrpdFallbackApnList(QmiService* pQmiService);
extern void RegisterWdsEventRegister(QmiService* pQmiService, pack_wds_SLQSWdsSetEventReport_t *pReqArg);
extern void RegisterWdsIndicationRegister(QmiService* pQmiService, pack_wds_indication_register_t   *pIndicationRegister);
extern void GetDefaultProfileId(QmiService* pQmiService, pack_wds_GetDefaultProfileNum_t *pGetDefaultProfileNum);
extern void SetDefaultProfileId(QmiService* pQmiService, pack_wds_SetDefaultProfileNum_t *pSetDefaultProfileNum);
extern void Get3GPPConfigItems(QmiService* pQmiService);
extern void Set3GPPConfigItems(QmiService* pQmiService, pack_wds_SLQSSet3GPPConfigItem_t *pSet3GPPConfigItem);
extern void SetDownlinkThroughputReportPeriod(QmiService* pQmiService, pack_wds_SetDLThroughputReportPeriod_t *pSetDLThroughputReportPeriod);
extern void QueryDownlinkThroughputReportingStatus(QmiService* pQmiService);
extern void GetCurrentChannelRate(QmiService* pQmiService);

static CtlService s_CtlService;
static QmiService s_WdsService;
static QmuxTransport s_Transport;

//////////////////////////////////////////////////////////
static void PrintPrompt();
static bool PrintInvalidArguments();
static void PrintGetKeepAliveOptions();
static void PrintGetResetAndModifyProfileInfo();
static void PrintGetApnOpReservedPCO();
static void PrintGetApnMsisdnIfo();
static void PrintDeleteAllProfilesInfo();
static void PrintSetEhrpdFallbackApn();
static void PrintSetModemAssitedKAStartInfo();
static void PrintSetModemAssitedKAStartInfoOptions();
static void PrintGetModemAssitedKAStartInfo();
static void PrintRegisterWdsEventReport();
static void PrintIndicationRegister();
static void PrintGetDefaultProfileIdInfo();
static void PrintSetDefaultProfileIdInfo();
static void PrintSet3GPPConfigItems();
static void PrintSetDownlinkThroughputReportPeriod();
static uint32_t ExtractValue(char *pChar);

enum AppStateType g_AppState = AppUninitialized;
static bool g_keepAliveDataSession = false;

#define DEVICE_PATH_MAX 256
char g_DevicePath[DEVICE_PATH_MAX]={0};
uint32_t g_keep_alive_type = 0;

bool g_PrintVersion = false;
int g_mode = QMUX_INTERFACE_UNKNOWN;

#define APPNAME "lite-qmi-wds"
#define VERSION "1.0.2111.0"

///////////////////////////////////////////////////
static void WdsIndicationCallback(uint8_t* qmiPacket, uint16_t qmiPacketSize, void* pIndicationCallbackContext)
{
	(void)pIndicationCallbackContext;

	unpack_qmi_t rsp_ctx;
	printf (ANSI_COLOR_YELLOW);
	printf("<< receiving wds indication: %s\n", helper_get_resp_ctx(eWDS, qmiPacket, qmiPacketSize, &rsp_ctx));
	printf("msgid 0x%x, type:%x\n", rsp_ctx.msgid,rsp_ctx.type);
	printf (ANSI_COLOR_RESET);
	wds_indication_handler(rsp_ctx.msgid, qmiPacket,  qmiPacketSize);
}

static void PrintGetKeepAliveOptions()
{
    fprintf( stderr, "\nPlease select Keep-Alive data session option"\
                        "  Or Press<Enter> to go to previous menu:\n"\
                        "  1. Enable Keep-Alive data session\n"\
                        "  2. Disable Keep-Alive data session\n" );
    fprintf( stderr, "Option:" );
    g_AppState = AppGetKeepAliveOption;
}

static void PrintGetResetAndModifyProfileInfo()
{
    fprintf( stderr, "\nPlease type profile ID and profiles type (3GPP=0, 3GPP2=1, EPC=2)"\
                        "  Or Press<Enter> to go to previous menu:\n");
    fprintf( stderr, "ex. ID=1 Type=2\n" );
    g_AppState = AppGetResetProfileInfo;
}

static void PrintGetApnOpReservedPCO()
{
    fprintf( stderr, "\nPlease enter APN name:\n"\
                        "  Or Press<Enter> for default:\n");
    fprintf( stderr, "APN Name: " );
    g_AppState = AppGetApnOpResPcoList;
}

static void PrintGetApnMsisdnIfo()
{
    fprintf( stderr, "\nPlease enter APN name:\n"\
                        "  Or Press<Enter> for default:\n");
    fprintf( stderr, "APN Name:" );
    g_AppState = AppGetApnMsisdnInfo;
}

static void PrintDeleteAllProfilesInfo()
{
    fprintf( stderr, "\nPlease enter profile type mask (3GPP=1, 3GPP2=2, 3GPP&3GPP2=3) "\
                        "  Or Press<Enter> to go to previous menu:\n");
    fprintf( stderr, "Option:" );
    g_AppState = AppGetDeleteAllProfilesInfo;
}

static void PrintSetEhrpdFallbackApn()
{
    fprintf( stderr, "\nPlease enter APN name\n");
    fprintf( stderr, "APN name:" );
    g_AppState = AppSetEhrpdFallbackApnList;
}

static void PrintSetModemAssitedKAStartInfo()
{
    fprintf( stderr, "\nPlease enter Keep-Alive type (TYPE_NAT=0) "\
                        "  Or Press<Enter> to go to previous menu:\n");
    fprintf( stderr, "Keep-Alive type:" );
    g_AppState = AppSetModemAssistedKaStart;
}

static void PrintSetModemAssitedKAStartInfoOptions()
{
    fprintf( stderr, "\nPlease enter at least one of the optional parameter:\n");
    fprintf( stderr, "Timer-value, APN-name, Dest-ipv4-address, Source-ipv4-address, Dest-port, Source-port\n" );
    fprintf( stderr, "ex. Timer-value=xx, APN-name=xx, Dest-ipv4-address=xx, \n" );
    fprintf( stderr, "    Source-ipv4-address=xx, Dest-port=xx, Source-port=xx\n\n" );

    g_AppState = AppSetModemAssistedKaStartOptions;
}

static void PrintGetModemAssitedKAStartInfo()
{
    fprintf( stderr, "\nPlease enter Keep-Alive handle \n");
    fprintf( stderr, "Keep-Alive handle:" );
    g_AppState = AppSetModemAssistedKaStop;
}

static void PrintRegisterWdsEventReport()
{
    fprintf( stderr, "\nPlease enter at least one of the optional parameter:\n");
    fprintf( stderr, "Current Channel Rate Indicator State Period(Optional) (0: Do not report 1: Reort) CCRI=0|1\n" );
    fprintf( stderr, "Transfer Statistics Indicator(Optional) (0: Do not report 1: Other - Period between reports (in seconds) ) TSI=0|x\n" );
    fprintf( stderr, "Transfer Statistics Indicator Mask(Optional) (0x1:Tx packets OK 0x2:Rx packets OK 0x4:Tx packet errors 0x8:Rx packet errors 0x10:Tx overflows 0x20:Rx overflows 0x40:Tx bytes OK 0x80:Rx bytes OK 0x100:Tx packets dropped 0x200:Rx packets dropped ) TSIM=\n" );
    fprintf( stderr, "Data Bearer Technology Indicator(Optional) (0: Do not report 1: Reort) DBTI=0|1\n" );
    fprintf( stderr, "Dormancy Status indicator(Optional) (0: Do not report 1: Reort) DSI=0|1\n" );
    fprintf( stderr, "MIP Status Indicator(Optional) (0: Do not report 1: Reort) MSI=0|1\n" );
    fprintf( stderr, "Data Call Status Change Indicator(Optional) (0: Do not report 1: Reort) DCSI=0|1\n" );
    fprintf( stderr, "Current Preferred Data System Indicator(Optional) (0: Do not report 1: Reort) CPDSI=0|1\n" );
    fprintf( stderr, "EV-DO Page Monitor Period Change Indicator(Optional) (0: Do not report 1: Reort) EPMCI=0|1\n" );
    fprintf( stderr, "Uplink Flow Control Indicator(Optional) (0: Do not report 1: Reort) UFCI=0|1\n" );
    fprintf( stderr, "Additional PDN Filters Removal Indicator(Optional) (0: Do not report 1: Reort) APFRI=0|1\n" );
    fprintf( stderr, "Data Bearer Technology Extended Indicator(Optional) (0: Do not report 1: Reort) DBTXI=0|1\n" );
    fprintf( stderr, "Dormancy Result Indicator(Optional) (0: Do not report 1: Reort) DRI=0|1\n" );
    
    fprintf( stderr, "ex. CCRI=1 TSI=1 TSIM=0x08|0x100|0x80 DSI=1 MSI=0 DCSI=0 CPDSI=1 EPMCI=0 UFCI=1 APFRI=1 DBTXI=1 DRI=1\n" );
    fprintf( stderr, "WDS Event Report Options:  " );
    g_AppState = AppSetWdsEventReport;
}


static void PrintIndicationRegister()
{
    fprintf( stderr, "\nPlease enter at least one of the optional parameter:\n");
    fprintf( stderr, "Service status indication (Optional) (0- Do not suppress 1- Suppress) SSI=0|1\n" );
    fprintf( stderr, "LTE attach PDN list (Optional) (0- Do not report 1- Report changed LTE attach PDN list) LAP=0|1\n" );
    fprintf( stderr, "Report Profile Changes (Optional) (0- Do not report 1- Report profile changed events) RPC=0|1\n" );
    fprintf( stderr, "Report APN List in Roaming (Optional) (0- Do not report 1- Report the list of APNs in Roaming) AAL=0|1\n" );
    fprintf( stderr, "Report APN parameter change information (Optional) (0- Do not report 1- Report APN parameter change) AAP=0|1\n" );
    fprintf( stderr, "Report LTE attach parameters (Optional) (0- Do not report 1- Report LTE attach parameters) RLA=0|1\n" );
    fprintf( stderr, "APN for which the client is interested in receiving changes in operator PCO change information: (Optional) () APN=abc|1\n" );
    fprintf( stderr, "APN for which the client is interested in receiving changes in MSISDN information (Optional) () APNM=0|1\n" );
    fprintf( stderr, "Report bearer type information (Optional) (0- Do not report 1- Report bearer type information) RBT=0|1\n" );
    fprintf( stderr, "Report Throttled PDN Information (Optional) (0- Do not report 1- Report throttled PDN information) RTP=0|1\n" );
    
    fprintf( stderr, "ex. SSI=1 LAP=1 RPC=1 AAL=1 AAP=1 RLA=1 APN=swi3.ca.apn APNM=xxx RBT=1 RTP=1\n" );
    fprintf( stderr, "WDS Indication Register Options:  " );

    g_AppState = AppWdsIndicationRegister;
}

static void PrintGetDefaultProfileIdInfo()
{
    fprintf( stderr, "\nPlease enter profile type and profile family:\n");
    fprintf( stderr, "Profile type (0: 3GPP 1: 3GPP2 2: EPC) PT=0|1|2\n" );
    fprintf( stderr, "Profile family (0: Embedded 1: Tethered) PF=0|1\n" );
    fprintf( stderr, "ex. PT=0 PF=1\n\n" );

    g_AppState = AppWdsGetDefaultProfileId;
}

static void PrintSetDefaultProfileIdInfo()
{
    fprintf( stderr, "\nPlease enter profile type, profile family, and Profile number to be set as default profile:\n");
    fprintf( stderr, "Profile type (0: 3GPP 1: 3GPP2 2: EPC) PT=0|1|2\n" );
    fprintf( stderr, "Profile family (0: Embedded 1: Tethered) PF=0|1\n" );
    fprintf( stderr, "Profile ID PID=xx\n" );
    fprintf( stderr, "ex. PT=0 PF=0 PID=2\n\n" );

    g_AppState = AppWdsSetDefaultProfileId;
}

static void PrintSet3GPPConfigItems()
{
    fprintf( stderr, "\nPlease enter one or more 3GPP Configuration Items:\n");
    fprintf( stderr, "Profile List (optional) (1 to 4 profiles) PA=x,xx,x,x\n" );
    fprintf( stderr, "Always connect default PDN (optional)(0: disabled 1: enabled) AC=0|1\n" );
    fprintf( stderr, "3GPP Release (optional)(0: Release99 1: Release5 2: Release6 3: Release7 4: Release8 5: Release9 6: Release10 7: Release11) GR=0|1|2|3|4|5|6|7\n" );
    fprintf( stderr, "LTE attach profiles in order of decreasing priority (1-24 possible values) (optional)(1 to 56 profiles) LP=x,x,xx,x,...\n" );
    fprintf( stderr, "ex. PA=1,3 AC=1 GR=3 LP=1,12,24\n\n" );

    g_AppState = AppWdsSet3GPPConfigItems;
}

static void PrintSetDownlinkThroughputReportPeriod()
{
    fprintf( stderr, "\nPlease enter Set Downlink Throughput Report Period in milliseconds:\n");

    g_AppState = AppWdsSetSetDLThroughputRP;
}

static void PrintPrompt()
{
    printf("\n     1.  Set Keep-Alive data session mode"
           "\n     2.  Reset and modify profile"
           "\n     3.  Get APN Operator Reserved PCO List"
           "\n     4.  Get APN MSISDN Info"
           "\n     5.  Delete All Profiles"
           "\n     6.  Get Ehrpd Fallback APN List"
           "\n     7.  Set Ehrpd Fallback APN List"
           "\n     8.  Set Modem Assisted Keep-Alive Start"
           "\n     9.  Set Modem Assisted Keep_Alive Stop"
           "\n     10. Register WDS Event Report"
           "\n     11. Register WDS Indications"
           "\n     12. Get Default Profile ID"
           "\n     13. Set Default Profile ID"
           "\n     14. Get 3GPP Configuration Items"
           "\n     15. Set 3GPP Configuration Items"
           "\n     16. Set Downlink Throughput Report Period"
           "\n     17. Get Downlink Throughput Reporting Status"
           "\n     18. Get Current Channel rate"
           "\n     (q)uit to exit: ");
    fflush(stdout);
}

static bool PrintInvalidArguments()
{
    printf("\n     Invalid arquments."
           "\n     Press<Enter> to go to previous menu.");
    fflush(stdout);
    return true;
}

static uint32_t ExtractValue(char *pChar)
{
    uint32_t value = 0;
    char *pCh = strchr(pChar, ',');
    if (pCh == NULL)
        pCh = strchr(pChar, ' ');
    if (!pCh)
    {
        value = atoi(pChar);
    }
    else
    {
        char szBuffer[255] = {0};
		StrCpy(szBuffer, pChar);
        value = atoi(szBuffer);
    }
    return value;
}

static void ExecuteWdsTestCaseSelection(void)
{
	char str_return[MAX_USER_INPUT_SIZE];
	char *pChar = NULL;
	PrintPrompt();
	
	while (1)
	{
		fflush(stdin);
		memset (str_return,0,MAX_USER_INPUT_SIZE);

		while (fgets(str_return, MAX_USER_INPUT_SIZE, stdin) == NULL);

		switch(g_AppState)
		{
			case AppCommandExecuted:
				g_AppState = AppRunning;
				PrintPrompt();
			break;
			case AppGetApnOpResPcoList:
            case AppGetApnMsisdnInfo:
			break;
			default:
			if (!strcmp(str_return, "\n"))
			{
				g_AppState = AppRunning;
				PrintPrompt();
			}
			break;
		}

		size_t s = strlen(str_return);
		if (s > 0)
			str_return[s - 1] = 0;
		else
			str_return[0] = 0;

		if (!strcmp(str_return, "q"))
		{
			printf("quitting...\n");
			break;
		} else if (!strcmp(str_return, "d"))
		{
			PrintPrompt();
		} 
        if (g_AppState == AppGetKeepAliveOption)
        {
            g_keepAliveDataSession = !(strcmp(str_return, "1")) ? true : false;
            SetKeepAliveDataSession(&s_WdsService, g_keepAliveDataSession);                
            g_AppState = AppCommandExecuted;
        }
        else if (g_AppState == AppGetResetProfileInfo)
        {
            uint8_t resetProfileID = 0xFF;
            uint8_t resetProfileType = 0xFF;

            pChar = strstr(str_return, "ID=");
            if (pChar)
            {
                pChar += 3;
                resetProfileID = atoi( pChar );
            }
            pChar = strstr(str_return, "Type=");           
            if (pChar)
            {
                pChar += 5;
                resetProfileType = atoi( pChar );
            }
            if (resetProfileID == 0xFF || resetProfileType == 0xFF)
            {
                PrintInvalidArguments();
            }
            else
            {
                wds_profileInfo  *pProfile = NULL; //TBD
                ResetAndModifyProfileSettings(&s_WdsService, resetProfileID, resetProfileType, pProfile);
            }
            
            g_AppState = AppCommandExecuted;
        }
        else if (g_AppState == AppGetApnOpResPcoList)
        {
            GetApnOpReservedPcoList(&s_WdsService, str_return);
            g_AppState = AppCommandExecuted;
        }
        else if (g_AppState == AppGetApnMsisdnInfo)
        {
            GetApnMsisdnInfo(&s_WdsService, str_return);
            g_AppState = AppCommandExecuted;
        }
        else if (g_AppState == AppGetDeleteAllProfilesInfo)
        {
            uint64_t  ProfileTypeMask =  atoi(str_return);
            uint64_t  *pProfilePersistenceMask = NULL;
            uint64_t  *pProfileClientTypeMask = NULL;
            DeleteAllProfiles(&s_WdsService, ProfileTypeMask, pProfilePersistenceMask, 
                                                    pProfileClientTypeMask);
            g_AppState = AppCommandExecuted;
        }
        else if (g_AppState == AppSetEhrpdFallbackApnList)
        {
            uint8_t fallback_apn_name_list_len = 1;
            WdsEhrpdFallbackApn_t wdsEhrpdFallbackApn;
            memset(&wdsEhrpdFallbackApn, 0, sizeof(WdsEhrpdFallbackApn_t));
            wdsEhrpdFallbackApn.apn_name_len = strlen(str_return);
			if (wdsEhrpdFallbackApn.apn_name_len >= sizeof(wdsEhrpdFallbackApn.apn_name))
				wdsEhrpdFallbackApn.apn_name_len = sizeof(wdsEhrpdFallbackApn.apn_name) - 1;
			StrNCpy(wdsEhrpdFallbackApn.apn_name, str_return, wdsEhrpdFallbackApn.apn_name_len);

            SetEhrpdFallbackApnList(&s_WdsService, fallback_apn_name_list_len, &wdsEhrpdFallbackApn);
            g_AppState = AppCommandExecuted;
        }
        else if (g_AppState == AppSetModemAssistedKaStart)
        {
            g_keep_alive_type = atoi(str_return);
            PrintSetModemAssitedKAStartInfoOptions();
        }
        else if(g_AppState == AppSetModemAssistedKaStartOptions)
        {
            if (strlen(str_return) == 0)
            {
                PrintInvalidArguments();
            }
            else
            {    
                pack_wds_modem_assisted_ka_start_t kaParam;
                memset(&kaParam, 0, sizeof(pack_wds_modem_assisted_ka_start_t));
                kaParam.keep_alive_type = g_keep_alive_type;

                uint32_t	Timer_value = 0;
                uint32_t	Dest_ipv4_address = 0;
                uint32_t	Source_ipv4_address = 0;
                uint16_t	Dest_port = 0;
                uint16_t	Source_port = 0;
                uint8_t		apn_name_len = 0;
                char		Apn_Name[255] = {0};
        
                pChar = strstr(str_return, "Timer-value=");
                if (pChar)
                {
                    pChar += 12;
                    Timer_value = ExtractValue( pChar );
                    kaParam.pTimer_value = &Timer_value; 
                }
                pChar = strstr(str_return, "APN-name=");
                if (pChar)
                {
                    pChar += 9;

                    char *pCh = strchr(pChar, ',');
                    if (pCh == NULL)
                        pCh = strchr(pChar, ' ');
                    if (!pCh)
                    {
						StrCpy(Apn_Name, pChar);
                    }
                    else
                    {
						StrNCpy(Apn_Name, pChar, pCh - pChar);
                    }
                    apn_name_len = strlen(Apn_Name); 
                    if (apn_name_len > 0)
                    {
                        kaParam.apn_name_len = &apn_name_len;
                        kaParam.pApn_Name = Apn_Name;
                    }
                }
                pChar = strstr(str_return, "Dest-ipv4-address=");
                if (pChar)
                {
                    pChar += 18;
                    Dest_ipv4_address = ExtractValue( pChar );
                    kaParam.pDest_ipv4_address = &Dest_ipv4_address;
                }
                pChar = strstr(str_return, "Source-ipv4-address=");
                if (pChar)
                {
                    pChar += 20;
                    Source_ipv4_address = ExtractValue( pChar );
                    kaParam.pSource_ipv4_address = &Source_ipv4_address; 
                }
                pChar = strstr(str_return, "Dest-port=");
                if (pChar)
                {
                    pChar += 10;
                    Dest_port = ExtractValue( pChar );
                    kaParam.pDest_port = &Dest_port; 
                }
                pChar = strstr(str_return, "Source-port=");
                if (pChar)
                {
                    pChar += 12;
                    Source_port = ExtractValue( pChar );
                    kaParam.pSource_port = &Source_port; 
                }

                //At least one optional parameter must be set.
                if (kaParam.pTimer_value || kaParam.pApn_Name || kaParam.pDest_ipv4_address || 
                    kaParam.pSource_ipv4_address || kaParam.pDest_port || kaParam.pSource_port)
                {
                    SetModemAssistedKaStart(&s_WdsService, &kaParam);
                }
                else
                {
                    PrintInvalidArguments();
                }                
            }

            g_AppState = AppCommandExecuted;
        }
        else if (g_AppState == AppSetModemAssistedKaStop)
        {
            uint32_t KeepAliveHandle = atoi(str_return);
            SetModemAssistedKaStop(&s_WdsService, KeepAliveHandle);
            g_AppState = AppCommandExecuted;
        }
        else if(g_AppState == AppSetWdsEventReport)
        {
            if (strlen(str_return) == 0)
            {
                PrintInvalidArguments();
            }
            else{
                //CCRI=1 TSI=1 TSIM=0x08|0x100|0x100 DSI=1 MSI=0 DCSI=0 CPDSI=1 EPMCI=0 UFCI=1 APFRI=1 DBTXI=1 DRI=
                uint8_t CCRI, DSI, MSI, DCSI, CPDSI, EPMCI, UFCI, APFRI, DBTXI, DRI;

                char *pCCRI = strstr(str_return, "CCRI=");
                char *pTSI = strstr(str_return, "TSI");
                char *pTSIM = strstr(str_return, "TSIM");
                char *pDSI = strstr(str_return, "DSI");
                char *pMSI = strstr(str_return, "MSI");
                char *pDCSI = strstr(str_return, "DCSI");
                char *pCPDSI = strstr(str_return, "CPDSI");
                char *pEPMCI = strstr(str_return, "EPMCI");
                char *pUFCI = strstr(str_return, "UFCI");                
                char *pAPFRI = strstr(str_return, "APFRI");
                char *pDBTXI = strstr(str_return, "DBTXI");
                char *pDRI = strstr(str_return, "DRI");

                pack_wds_SLQSWdsSetEventReport_t reqArg;
                memset (&reqArg, 0, sizeof(pack_wds_SLQSWdsSetEventReport_t));

                if (pCCRI)
                {
                    pCCRI += 5;
                    CCRI = atoi(pCCRI);
                    reqArg.pCurrChannelRateInd = &CCRI;
                }
                if (pTSI && pTSIM)
                {
                    wds_TrStatInd   TransferStatInd;
                    pTSI += 4;
                    pTSIM += 5;
                    TransferStatInd.statsPeriod = atoi(pTSI);

                    uint32_t mask = 0x00;
                    char szMask[20] = {0};
                    char *pPos1 = pTSIM;
                    char *pPos2 = strchr(pPos1, ' ');
                    if (pPos2)
                    {
                        *pPos2 = 0;

                        while (pPos1 < (pTSIM + strlen(pTSIM)))
                        {
                            pPos2 = strchr(pPos1, '|');
                            if (pPos2 == NULL)
                            {
                                sscanf( pPos1, "%x" , &mask);
                                TransferStatInd.statsMask |= mask; 
                                break;
                            }
                            StrNCpy(szMask, pPos1, (pPos2 - pPos1));
                            sscanf( szMask, "%x", &mask);
                            TransferStatInd.statsMask |= mask; 
                            pPos1 = pPos2 + 1;
                        }
                        reqArg.pTransferStatInd = &TransferStatInd;
                    }
                }            
                if (pDSI)
                {
                    pDSI += 4;
                    DSI = atoi(pDSI);
                    reqArg.pDormancyStatusInd = &DSI;
                }
                if (pMSI)
                {
                    pMSI += 4;
                    MSI = atoi(pMSI);
                    reqArg.pMIPStatusInd = &MSI;
                }
                if (pDCSI)
                {
                    pDCSI += 5;
                    DCSI = atoi(pDCSI);
                    reqArg.pDataCallStatusChangeInd = &DCSI;
                }
                if (pCPDSI)
                {
                    pCPDSI += 6;
                    CPDSI = atoi(pCPDSI);
                    reqArg.pCurrPrefDataSysInd = &CPDSI;
                }
                if (pEPMCI)
                {
                    pEPMCI += 6;
                    EPMCI = atoi(pEPMCI);
                    reqArg.pEVDOPageMonPerChangeInd = &EPMCI;
                }                
                if (pUFCI)
                {
                    pUFCI += 5;
                    UFCI = atoi(pUFCI);
                    reqArg.pUlFlowControlInd = &UFCI;
                }
                if (pAPFRI)
                {
                    pAPFRI += 6;
                    APFRI = atoi(pAPFRI);
                    reqArg.pReportAddPdnFiltersRemoval = &APFRI;
                }
                if (pDBTXI)
                {
                    pDBTXI += 6;
                    DBTXI = atoi(pDBTXI);
                    reqArg.pDataBearerTechExtInd = &DBTXI;
                }
                if (pDRI)
                {
                    pDRI += 4;
                    DRI = atoi(pDRI);
                    reqArg.pDormancyResultIndicator = &DRI;
                }
                RegisterWdsEventRegister(&s_WdsService, &reqArg);
            }
            g_AppState = AppCommandExecuted;
        }
        else if(g_AppState == AppWdsIndicationRegister)
        {
            if (strlen(str_return) == 0)
            {
                PrintInvalidArguments();
            }
            else
            {
                //SSI=1 LAP=1 RPC=1 AAL=1 AAP=1 RLA=1 APN=swi3.ca.apn APNM=xxx RBT=1 RTP=1
                uint8_t SSI, LAP, RPC, AAL, AAP, RLA, RCP, RCM, RBT, RTP, ApnNameLen, ApnNameLenMsisdn;
                char szAPN[100];
                char szApnMsisdn[100];
                memset(szAPN, 0, 100*sizeof(char));
                memset(szApnMsisdn, 0, 100*sizeof(char));
                char *pSSI = strstr(str_return, "SSI=");
                char *pLAP  = strstr(str_return, "LAP=");
                char *pRPC  = strstr(str_return, "RPC=");
                char *pAAL  = strstr(str_return, "AAL=");
                char *pAAP  = strstr(str_return, "AAP=");
                char *pRLA  = strstr(str_return, "RLA=");
                char *pAPN  = strstr(str_return, "APN=");
                char *pAPNM  = strstr(str_return, "APNM=");
                char *pRBT  = strstr(str_return, "RBT=");
                char *pRTP  = strstr(str_return, "RTP=");

                pack_wds_indication_register_t   sIndicationRegister;
                memset(&sIndicationRegister, 0, sizeof(pack_wds_indication_register_t));
                if (pSSI)
                {
                    pSSI += 4;
                    SSI = atoi(pSSI);
                    sIndicationRegister.pSupperesPktSrvcInd = &SSI;
                }
                if (pLAP)
                {
                    pLAP += 4;
                    LAP = atoi(pLAP);
                    sIndicationRegister.pReportLteAttachPdnListChange = &LAP;
                }
                if (pRPC)
                {
                    pRPC += 4;
                    RPC = atoi(pRPC);
                    sIndicationRegister.pReportProfileChangeEvents = &RPC;
                }
                if (pAAL)
                {
                    pAAL += 4;
                    AAL = atoi(pAAL);
                    sIndicationRegister.pReportRoamingApnList = &AAL;
                }
                if (pAAP)
                {
                    pAAP += 4;
                    AAP = atoi(pAAP);
                    sIndicationRegister.pReportApnParamChangeInfo = &AAP;
                }
                if (pRLA)
                {
                    pRLA += 4;
                    RLA = atoi(pRLA);
                    sIndicationRegister.pReportLteAttachParams = &RLA;
                }
                if (pAPN)
                {   
                    pAPN += 4;
                    char *pCh = strchr(pAPN, ' ');
                    if (pCh)
                    {
		                StrNCpy(szAPN, pAPN, pCh - pAPN);

                        sIndicationRegister.pApnName = szAPN;
                        ApnNameLen = strlen(szAPN);
                        sIndicationRegister.pApnNameLen = &ApnNameLen; 

                        RCP = 1;
                        sIndicationRegister.pReportOpResevedPcoListChange = &RCP;
                    }
                }
                if (pAPNM)
                {
                    pAPNM += 5;
                    char *pCh = strchr(pAPNM, ' ');
                    if (pCh)
                    {
		                StrNCpy(szApnMsisdn, pAPN, pCh - pAPNM);

                        sIndicationRegister.pApnNameMsisdn = szApnMsisdn;
                        ApnNameLenMsisdn = strlen(szApnMsisdn);
                        sIndicationRegister.pApnNameMsisdnLen = &ApnNameLenMsisdn; 
                        RCM = 1;
                        sIndicationRegister.pReportMsisdnInfoChange = &RCM;
                    }
                }
                if (pRBT)
                {
                    pRBT += 4;
                    RBT = atoi(pRBT);
                    sIndicationRegister.pReportBearerType = &RBT;
                }
                if (pRTP)
                {
                    pRTP += 4;
                    RTP = atoi(pRTP);
                    sIndicationRegister.pReportPdnThrottleInfo = &RTP;
                }
                
                RegisterWdsIndicationRegister(&s_WdsService, &sIndicationRegister);
            }
            g_AppState = AppCommandExecuted;
        }
        else if (g_AppState == AppWdsGetDefaultProfileId)
        {
            char *pPT = strstr(str_return, "PT=");
            char *pPF  = strstr(str_return, "PF=");
            if (pPT == NULL || pPF == NULL)
            {
                PrintInvalidArguments();
            }
            else
            {
                pack_wds_GetDefaultProfileNum_t  sGetDefaultProfileNum;
                memset (&sGetDefaultProfileNum, 0, sizeof(pack_wds_GetDefaultProfileNum_t));
                pPT += 3;
                pPF += 3;
                sGetDefaultProfileNum.type = atoi(pPT);
                sGetDefaultProfileNum.family = atoi(pPF);
                if (sGetDefaultProfileNum.type > 2 || sGetDefaultProfileNum.family > 1)
                {
                    PrintInvalidArguments();
                }
                else
                {
                    GetDefaultProfileId(&s_WdsService, &sGetDefaultProfileNum);
                }
            }

            g_AppState = AppCommandExecuted;
        }
        else if (g_AppState == AppWdsSetDefaultProfileId)
        {
            char *pPT = strstr(str_return, "PT=");
            char *pPF  = strstr(str_return, "PF=");
            char *pPID  = strstr(str_return, "PID=");
            if (pPT == NULL || pPF == NULL || pPID == NULL)
            {
                PrintInvalidArguments();
            }
            else
            {
                pack_wds_SetDefaultProfileNum_t  sSetDefaultProfileNum;
                memset (&sSetDefaultProfileNum, 0, sizeof(pack_wds_SetDefaultProfileNum_t));
                pPT += 3;
                pPF += 3;
                pPID += 4;
                sSetDefaultProfileNum.type = atoi(pPT);
                sSetDefaultProfileNum.family = atoi(pPF);
                sSetDefaultProfileNum.index = atoi(pPID);
                if (sSetDefaultProfileNum.type > 2 || sSetDefaultProfileNum.family > 1)
                {
                    PrintInvalidArguments();
                }
                else
                {
                    SetDefaultProfileId(&s_WdsService, &sSetDefaultProfileNum);
                }
            }

            g_AppState = AppCommandExecuted;
        }
        else if (g_AppState == AppWdsSet3GPPConfigItems)
        {
            //PA=1,3 AC=1 GR=3 LP=1,12,24
            char *pPA = strstr(str_return, "PA=");
            char *pAC = strstr(str_return, "AC=");
            char *pGR = strstr(str_return, "GR=");
            char *pLP = strstr(str_return, "LP=");

            if (pPA == NULL && pAC == NULL && pGR == NULL && pLP == NULL)
            {
                PrintInvalidArguments();
            }
            else
            {
                pack_wds_SLQSSet3GPPConfigItem_t sSet3GPPConfigItem;
                memset (&sSet3GPPConfigItem, 0, sizeof(pack_wds_SLQSSet3GPPConfigItem_t));
                uint16_t ProfileList[4] = {0};
                uint8_t  nDefaultPDNEnabled = 0;
                uint8_t  n3gppRelease = 0;
                uint16_t LTEAttachProfileList[56] = {0};

                sSet3GPPConfigItem.pLTEAttachProfile = NULL;
                if (pPA)
                {   
                    int idx = 0;
                    uint16_t nProfileId = 0;
                    char szProfileIds[20] = {0};
                    pPA += 3;
                    char *pCommaPos = NULL;
                    char *pSpacePos = strchr(pPA, ' ');

                    if (pSpacePos != NULL)
                    {
                        if ((pSpacePos - pPA) < 12)
                            StrNCpy(szProfileIds, pPA, (pSpacePos - pPA));
                    }
                    else if((pSpacePos - pPA) < 12)
                    {
                        StrCpy(szProfileIds, pPA);
                    }

                    pPA = szProfileIds;
                    while (idx < 4)
                    {
                        pCommaPos = strchr(pPA, ',');
                        if (pCommaPos == NULL)
                        {
                            nProfileId = atoi(pPA); 
                            if (nProfileId < 25)
                            {
                                ProfileList[idx] = nProfileId;
                                ++idx;
                            }
                            break;
                        }
                        if ((pCommaPos - pPA) < 3)
                        {
                            nProfileId = atoi(pPA);
                            if (nProfileId < 25)
                            {
                                ProfileList[idx] = nProfileId;
                                ++idx;
                            }
                        }
                        pPA = pCommaPos + 1;
                    }
                    if (idx > 0)
                    {
                        sSet3GPPConfigItem.pProfileList = ProfileList;
                    }
                }
                if (pAC)
                {
                    pAC += 3;
                    nDefaultPDNEnabled = atoi(pAC);
                    sSet3GPPConfigItem.pDefaultPDNEnabled = &nDefaultPDNEnabled;
                }
                if (pGR)
                {
                    pGR += 3;
                    n3gppRelease = atoi(pGR);
                    sSet3GPPConfigItem.p3gppRelease = &n3gppRelease;
                }
                if (pLP)
                {
                    uint16_t nProfileId = 0;
                    int idx = 0;
                    char szProfileIds[500] = {0};
                    pLP += 3;
                    char *pCommaPos = NULL;
                    char *pSpacePos = strchr(pLP, ' '); 
                    if (pSpacePos != NULL)
                    {
                        if ((pSpacePos - pLP) < 500)
                            StrNCpy(szProfileIds, pLP, (pSpacePos - pLP));
                    }
                    else if((pSpacePos - pLP) < 500)
                    {
                        StrCpy(szProfileIds, pLP);
                    }

                    while (idx < 56)
                    {
                        pCommaPos = strchr(pLP, ',');
                        if (pCommaPos == NULL)
                        {
                            nProfileId = atoi( pLP); 
                            if (nProfileId < 25)
                            {
                                LTEAttachProfileList[idx] = atoi( pLP);
                                ++idx;
                            }
                            break;
                        }
                        if ((pCommaPos - pLP) < 3)
                        {
                            nProfileId = atoi( pLP);
                            if (nProfileId < 25)
                            {
                                LTEAttachProfileList[idx] = nProfileId;
                                ++idx;
                            }
                        }
                        pLP = pCommaPos + 1;
                    }
                    if (idx > 0)
                    {
                        sSet3GPPConfigItem.pLTEAttachProfileList = LTEAttachProfileList;
                        sSet3GPPConfigItem.LTEAttachProfileListLen = idx;
                    }
                }
                Set3GPPConfigItems(&s_WdsService, &sSet3GPPConfigItem);
            }

            g_AppState = AppCommandExecuted;
        }
        else if (g_AppState == AppWdsSetSetDLThroughputRP)
        {
            pack_wds_SetDLThroughputReportPeriod_t sSetDLThroughputReportPeriod;
            uint32_t downlink_throughput_report_period = atoi(str_return);
            sSetDLThroughputReportPeriod.pDownlink_throughput_report_period = &downlink_throughput_report_period;
            SetDownlinkThroughputReportPeriod(&s_WdsService, &sSetDLThroughputReportPeriod);
        }
		else if (g_AppState == AppRunning)
        {
            if (!strcmp(str_return, "1"))
            {
                PrintGetKeepAliveOptions();
            }
            else if (!strcmp(str_return, "2") || !strcmp(str_return, "Reset and modify profile"))
            {
                PrintGetResetAndModifyProfileInfo();
            }
            else if (!strcmp(str_return, "3") || !strcmp(str_return, "Get APN Operator Reserved PCO List"))
            {
                PrintGetApnOpReservedPCO();
            }
            else if (!strcmp(str_return, "4") || !strcmp(str_return, "Get APN MSISDN Info"))
            {
                PrintGetApnMsisdnIfo();
            }
            else if (!strcmp(str_return, "5") || !strcmp(str_return, "Delete All Profiles"))
            {
                PrintDeleteAllProfilesInfo();
            }
            else if (!strcmp(str_return, "6") || !strcmp(str_return, "Get Ehrpd Fallback APN List"))
            {
                GetEhrpdFallbackApnList(&s_WdsService);
            }
            else if (!strcmp(str_return, "7") || !strcmp(str_return, "Set Ehrpd Fallback APN List"))
            {
                PrintSetEhrpdFallbackApn();
            }
            else if (!strcmp(str_return, "8") || !strcmp(str_return, "Set Modem Assisted Keep-Alive Start"))
            {
                PrintSetModemAssitedKAStartInfo();
            }
            else if (!strcmp(str_return, "9") || !strcmp(str_return, "Set Modem Assisted Keep_Alive Stop"))
            {
                PrintGetModemAssitedKAStartInfo();
            }
            else if (!strcmp(str_return, "10") || !strcmp(str_return, "Register WDS Event Report"))
            {
                PrintRegisterWdsEventReport();
            }
            else if (!strcmp(str_return, "11") || !strcmp(str_return, "Register WDS Indications"))
            {
                PrintIndicationRegister();
            }
            else if (!strcmp(str_return, "12") || !strcmp(str_return, "Get Default Profile ID"))
            {
                PrintGetDefaultProfileIdInfo();
            }
            else if (!strcmp(str_return, "13") || !strcmp(str_return, "Set Default Profile ID"))
            {
                PrintSetDefaultProfileIdInfo();
            }
            else if (!strcmp(str_return, "14") || !strcmp(str_return, "Get 3GPP Configuration Items"))
            {
                Get3GPPConfigItems(&s_WdsService);
            }
            else if (!strcmp(str_return, "15") || !strcmp(str_return, "Set 3GPP Configuration Items"))
            {
                PrintSet3GPPConfigItems();
            }
            else if (!strcmp(str_return, "16") || !strcmp(str_return, "Set Downlink Throughput Report Period"))
            {
                PrintSetDownlinkThroughputReportPeriod();
            }
            else if (!strcmp(str_return, "17") || !strcmp(str_return, "Get Downlink Throughput Reporting Status"))
            {
                QueryDownlinkThroughputReportingStatus(&s_WdsService);
            }
            else if (!strcmp(str_return, "18") || !strcmp(str_return, "Get Current Channel rate"))
            {
                GetCurrentChannelRate(&s_WdsService);
            }
        }	
	}

	return;
}

void PrintUsageManual()
{
    printf( "\r\n" );
    printf( "App usage: \r\n\r\n" );
    printf( "  -d  --device \n" );
    printf( "        absolute path to qmux device\n\n" );
    printf( "  -m  --mbim \n" );
    printf( "        Device is an MBIM interface (defaults to direct QMUX interface)\n\n" );
	printf( "  -q  --qmi\n");
	printf( "        Use direct QMUX interface (QMI over PCIe, or RmNet over USB)\n\n");
	printf( "  -r  --router\n");
	printf( "        Use QMUX Router\n\n");
	printf( "  -h  --help  \n" );
    printf( "        This option prints the usage instructions.\n\n" );
    printf( "  -V  --version  \n" );
    printf( "        This option prints app version.\n\n" );
}

const char * const s_ShortOptions = "d:hmqrV";

const struct option s_LongOptions[] = {
    {"version",  0, NULL, 'V'},
    {"help",   0, NULL, 'h'},
    {"device", 1, NULL, 'd'},
    {"mbim",   0, NULL, 'm'},
    {"qmi",    0, NULL, 'q'},
	{"router", 0, NULL, 'r'},
	{NULL,     0, NULL,  0 }       /* End of list */
};

static void ParseCommandLine( int argc, char **argv)
{
    int next_option;
    
    /* Parse the command line before doing anything else */
    do
    {
        /* Read the next option until there are no more */
        next_option = getopt_long( argc, argv,
                                   s_ShortOptions,
                                   s_LongOptions, NULL );

        switch( next_option )
        {
            case 'V':
                /* Print usage information */
                g_PrintVersion = true;
                break;
            case 'h':
                /* Print usage information */
                PrintUsageManual();
                exit (0);
                break;
            case 'd':
				StrCpy(g_DevicePath, optarg);
                break;
            case 'm':
                if (g_mode == QMUX_INTERFACE_UNKNOWN)
                    g_mode = QMUX_INTERFACE_MBIM;
                break;
            case 'q':
                if (g_mode == QMUX_INTERFACE_UNKNOWN)
                    g_mode = QMUX_INTERFACE_DIRECT;
                break;
			case 'r':
				if (g_mode == QMUX_INTERFACE_UNKNOWN)
					g_mode = QMUX_INTERFACE_ROUTER;
				break;
			case -1:
                /* Done with options list */
                break;
            default:
                exit(EXIT_FAILURE);
                break;
        }
    }
    while( next_option != -1 );
}

int main(int argc, char **argv)
{
	int ret = FAILURE;
    g_PrintVersion = false;

    ParseCommandLine(argc, argv);
    if (g_PrintVersion)
    {
        printf("\n%s v%s\n\n", APPNAME, VERSION);
       	if (argc == 2)
		    return 0;
    }

    if (OpenTransport(&s_Transport, g_DevicePath, sizeof(g_DevicePath), &g_mode, NULL,
		true, PrintUsageManual, argv[0], false) != 0)
		return 0;

	if (CtlService_Initialize(&s_CtlService, &s_Transport) != SUCCESS)
		printf("CtlService_Initialize failed\n");
	else
	{
		memset(&s_WdsService, 0, sizeof(QmiService));

        // We use the Ctl service to initialize the regular services because it knows how to 
        // acquire client IDs for them.

        // Use a do/while for easy exit. We have to clean up.
        do
        {
            // Infrastructure is now ready. Let's create some regular QMI services.

            ret = CtlService_InitializeRegularService(&s_CtlService, &s_WdsService, eWDS, WdsIndicationCallback, NULL);
            if (ret != SUCCESS)
            {
                printf("InitializeRegularService eWDS failed.\n");
                break;
            }
            g_AppState = AppRunning;
            
            ExecuteWdsTestCaseSelection();

        } while (0);		

        // Shut down.
        g_AppState = AppShuttingDown;

        CtlService_ShutDownRegularService(&s_CtlService, &s_WdsService);

        CtlService_ShutDown(&s_CtlService);
    }

	QmuxTransport_ShutDown(&s_Transport);
	
	return ret;
}
