/**
 * \ingroup liteqmi
 * \ingroup nas
 * 
 * \file swinas.h
 */
#ifndef __LITEQMI_SWINAS_H__
#define __LITEQMI_SWINAS_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "common.h"
#include <stdint.h>

#define MAX_RF_STATUS_INSTS     55

/**
 * \ingroup nas
 * 
 * pack function to get rfdev status
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 * 
 * @For_5G
 */

int pack_swinas_GetRfdevStatus(
    pack_qmi_t * pCtx,
    uint8_t * pReqBuf,
    uint16_t * pLen);

/**
 * \ingroup nas
 * 
 * This structure contains device RF status info
 *
 *  @param  inst_id
 *          - Instance ID
 * 
 *  @param  mfg_id
 *          - Manufacturer ID
 * 
 *  @param  prod_id
 *          - Product ID
 *
 *  @param  dev_status
 *          - 0: RF device is not connected or malfunctioning
 *          - 1: RF device present and functional
 * 
 *  @For_5G
 */
typedef struct {
    uint8_t     inst_id;
    uint16_t    mfg_id;
    uint16_t    prod_id;
    uint8_t     dev_status;
} rf_status_t;

/**
 * \ingroup nas
 * 
 * This structure contains RFC device status info
 * 
 *  @param  rf_init_status
 *          - RF initialization status
 *          - 0: RF initialization failed
 *          - 1: RF initialization completed.
 *               Refer to TLV RF Status for details
 * 
 *  @param  rf_status_num
 *          - Number of valid RF status instances included in rf_status array
 * 
 *  @param  rf_status
 *          - Array of RF status instances
 * 
 *  @For_5G
 */
typedef struct {
    uint8_t     rf_init_status;
    uint32_t    rf_status_num;
    rf_status_t rf_status[MAX_RF_STATUS_INSTS];
    swi_uint256_t ParamPresenceMask;
} unpack_swinas_GetRfdevStatus_t;

/**
 * \ingroup nas
 * 
 * unpack function to get rfdev status info
 * 
 * @param[in] pResp qmi response
 * @param[in] respLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 * 
 * @For_5G
 */

int unpack_swinas_GetRfdevStatus(
    uint8_t * pResp,
    uint16_t respLen,
    unpack_swinas_GetRfdevStatus_t * pOutput);

/**
 * \ingroup nas
 * 
 *  This structure contains the info for mmWave RFC configuration.
 *
 *  @param  hw_id
 *          - mmWave hardware ID
 *          - Use 0 in case of disabling mmWave
 * 
 *  @param  board_id
 *          - mmWave board ID
 *          - Use 0 in case of disabling mmWave
 *
 *  @For_5G
 *
 */
typedef struct
{
	uint16_t    hw_id;
    uint16_t    board_id;
} pack_swinas_SetMmwaveConfig_t;

typedef unpack_result_t  unpack_swinas_SetMmwaveConfig_t;

/**
 * \ingroup nas
 * 
 *  Pack function for mmWave configuration.
 * 
 *  @param[in,out] pCtx qmi request context
 *  @param[out] pReqBuf qmi request buffer
 *  @param[out] pLen qmi request length
 *  @param[in] pReqParam request prarmeters
 *
 *  @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 *  @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 *  @note   Device Supported: MC75xx, EM75xx and WP76xx.
 *
 *  @For_5G
 *
 */
int pack_swinas_SetMmwaveConfig(
	pack_qmi_t * pCtx,
	uint8_t * pReqBuf,
	uint16_t * pLen,
	pack_swinas_SetMmwaveConfig_t * pReqParam);

/**
 * \ingroup nas
 * 
 *  Unpack function for mmWave configuration.
 * 
 *  @param[in]   pResp       qmi response from modem
 *  @param[in]   respLen     qmi response length
 *  @param[out]  pOutput     unpacked response
 *
 *  @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 *  @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 *  @For_5G
 *
 */
int unpack_swinas_SetMmwaveConfig(
	uint8_t * pResp,
	uint16_t respLen,
	unpack_swinas_SetMmwaveConfig_t * pOutput);

/**
 * \ingroup swinas
 * 
 * This structure contains SWI NAS PDP context parameters info
 * 
 *  @param  cid
 *          - Connection ID
 *          - Specifies a particular PDP context definition
 * 
 *  @param  pdp_type
 *          - Specifies the type of packet data protocol:
 *          - X.25, IP, IPV6, IPV4V6, OSPIH, PPP, Non-IP, Ethernet.
 *          - Maximum 10 characters
 * 
 *  @param  APN
 *          - a logical name that is used to select the GGSN or the external packet data network
 *          - Maximum 100 characters
 * 
 *  @param PDP_addr
 *          - Identifies the MT in the address space applicable to the PDP
 *          - Maximum 40 characters
 * 
 *  @param d_comp 
 *          - Controls PDP data compression
 *           - values:
 *              - 0: off
 *              - 1: on
 *              - 2: V.42bis
 *              - 3: V.44
 * 
 *  @param h_comp
 *          - controls PDP header compression
 *           - values
 *              - 0: off
 *              - 1: on
 *              - 2: RFC1144
 *              - 3: RFC2507
 *              - 4: RFC3095
 * 
 *  @param IPv4AddrAlloc
 *          - Controls how the MT/TA requests to get the IPv4 address information
 *           - values:
 *              - 0: IPv4 address allocation through NAS signalling
 *              - 1: IPv4 address allocated through DHCP
 * 
 *  @param request_type
 *          - Indicates the type of PDP context activation request
 *           - values:
 *              - 0: PDP context is for new PDP context establishment or for handover from a non-3GPP access network.
 *              - 1: PDP context is for emergency (bearer) services.
 *              - 2: PDP context is for new PDP context establishment
 *              - 3: PDP context is for handover from a non-3GPP access network
 *              - 4: PDP context is for handover of emergency (bearer) services from a non-3GPP access network
 * 
 *  @param P_CSCF_discovery
 *          - Influences how the MT/TA requests to get the P-CSCF address
 *           - values:
 *              - 0: Preference of P-CSCF address discovery not influenced by +CGDCONT
 *              - 1: Preference of P-CSCF address discovery through NAS signaling
 *              - 2: Preference of P-CSCF address discovery through DHCP
 *
 *  @param IM_CN_Signalling_Flag_Ind
 *          - Indicates to the network whether the PDP context is for IM CN0x7568 subsystem-related signaling only or not
 *           - values:
 *              - 0: UE indicates that the PDP context is not for IM CN subsystem-related signalling only
 *              - 1: UE indicates that the PDP context is for IM CN subsystem-related signalling only
 * 
 *  @param NSLPI
 *          - Indicates the NAS signaling priority requested for this PDP context
 *           - values:
 *              - 0: indicates that this PDP context is to be activated with the value for the low priority indicator configured in the MT
 *              - 1: indicates that this PDP context is to be activated with the value for the low priority indicator set to "MS is not configured for NAS signalling low priority".
 *
 *  @param securePCO
 *          - Specifies if security protected transmission of PCO is requested or not.
 *           - values:
 *              - 0: Security protected transmission of PCO is not requested;
 *              - 1: Security protected transmission of PCO is requested.
 *
 *  @param IPv4_MTU_discovery
 *          - Influences how the MT/TA requests to get the IPv4 MTU size
 *           - values:
 *              - 0: Preference of IPv4 MTU size discovery not influenced by +CGDCONT
 *              - 1: Preference of IPv4 MTU size discovery through NAS signalling 
 * 
 *  @param Local_Addr_Ind
 *          - Indicates to the network whether the MS supports local IP address in TFTs
 *           - values:
 *              - 0: indicates that the MS does not support local IP address in TFTs
 *              - 1: indicates that the MS supports local IP address in TFTs
 * 
 *  @param Non_IP_MTU_discovery
 *          - Influences how the MT/TA requests to get the Non-IP MTU size
 *           - values
 *              - 0: Preference of Non-IP MTU size discovery not influenced by +CGDCONT
 *              - 1: Preference of Non-IP MTU size discovery through NAS signalling
 *
 *  @param Reliable_Data_Service
 *          - Indicates whether the UE is using Reliable Data Service for a PDN connection or not
 *           - values:
 *              - 0: Reliable Data Service is not being used for the PDN connection
 *              - 1: Reliable Data Service is being used for the PDN connection 
 * 
 *  @param SSC_mode
 *          - Indicates the session and service continuity (SSC) mode for the PDU session in 5GS
 * 
 *  @param S_NSSAI
 *          - S-NSSAI is associated with the PDU session for identifying a network slice.
 *          - Maximum 12 characters
 * 
 *  @param pref_access_type
 *          - Indicates the preferred access type for the PDU session in 5GS
 *           - values:
 *              - 0: indicates that the preferred access type is 3GPP access
 *              - 1: indicates that the preferred access type is non-3GPP access 
 *  @param RQoS_ind
 *          - Indicates whether the UE supports reflective QoS for the PDU session
 *           - values:
 *              - 0: indicates that reflective QoS is not supported for the PDU session
 *              - 1: indicates that reflective QoS is supported for the PDU session
 * 
 *  @param MH6_PDU
 *          - Indicates whether the UE supports IPv6 multi-homing for the PDU session
 *           - values:
 *              - 0: indicates that IPv6 multi-homing is not supported for the PDU session
 *              - 1: indicates that IPv6 multi-homing is supported for the PDU session

 *  @param  always_on_req
 *          - Indicates whether the UE requests to establish the PDU session as an always-on PDU session
 *          - 0: always-on PDU session is not requested 
 *          - 1: always-on PDU session is requested
 * 
 *  @param old_cid
 *          - Indicates the context identifier of the QoS flow of the default QoS rule of the SSC mode 2 or
 *          - SSC mode 3 PDU session where the network requests relocation of the PDU session anchor
 * 
 *  @For_5G
 */
typedef struct {
    uint16_t cid;
    char pdp_type[10];
    char APN[64];
    char PDP_addr[40];
    uint8_t d_comp;
    uint8_t h_comp;
    uint8_t IPv4AddrAlloc;
    uint8_t request_type;
    uint8_t P_CSCF_discovery;
    uint8_t IM_CN_Signalling_Flag_Ind;
    uint8_t NSLPI;
    uint8_t securePCO;
    uint8_t IPv4_MTU_discovery;
    uint8_t Local_Addr_Ind;
    uint8_t Non_IP_MTU_discovery;
    uint8_t Reliable_Data_Service;
    uint8_t SSC_mode;
    char S_NSSAI[12];
    uint8_t pref_access_type;
    uint8_t RQoS_ind;
    uint8_t MH6_PDU;
    uint8_t always_on_req;
    uint8_t old_cid;
    swi_uint256_t ParamPresenceMask;
} unpack_swinas_GetPdpContextParams_t;

/**
 * \ingroup nas
 * 
 * pack function to get PDP Profile Settings Ext Info
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 * 
 * @For_5G
 */

int pack_swinas_GetPDPContextParams(
    pack_qmi_t * pCtx,
    uint8_t * pReqBuf,
    uint16_t * pLen);

/**
 * \ingroup nas
 * 
 * unpack function to get PDP Profile Settings Ext info
 * 
 * @param[in] pResp qmi response
 * @param[in] respLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 * 
 * @For_5G
 */

int unpack_swinas_GetPDPContextParams(
    uint8_t * pResp,
    uint16_t respLen,
    unpack_swinas_GetPdpContextParams_t *pOutput);

typedef unpack_swinas_GetPdpContextParams_t pack_swinas_SetPdpContextParams_t;

/**
 * \ingroup nas
 * 
 *  Pack function for Set PDP Profile Settings Ext.
 * 
 *  @param[in,out] pCtx qmi request context
 *  @param[out] pReqBuf qmi request buffer
 *  @param[out] pLen qmi request length
 *  @param[in] pReqParam request prarmeters
 *
 *  @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 *  @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 *  @note   Device Supported: MC75xx, EM75xx and WP76xx.
 *
 *  @For_5G
 *
 */
int pack_swinas_SetPdpContextParams(
	pack_qmi_t * pCtx,
	uint8_t * pReqBuf,
	uint16_t * pLen,
	pack_swinas_SetPdpContextParams_t *pReqParam);

typedef unpack_result_t  unpack_swinas_SetPdpContextParams_t;

/**
 * \ingroup nas
 * 
 *  Unpack function for Set PDP Profile Settings Ext.
 * 
 *  @param[in]   pResp       qmi response from modem
 *  @param[in]   respLen     qmi response length
 *  @param[out]  pOutput unpacked response
 *
 *  @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 *  @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 *  @For_5G
 *
 */
int unpack_swinas_SetPdpContextParams(
	uint8_t * pResp,
	uint16_t respLen,
	unpack_swinas_SetPdpContextParams_t *pOutput);

/**
 * \ingroup nas
 * 
 * This structure contains SWI NAS Default NSSAI info
 * 
 *  @param  default_configured_nssai_length
 *          - Indicates the length in octets of the default configured NSSAI to be stored at the MT
 * 
 *  @param pDefault_configured_nssai
 *          - Indicates the default NSSAI
 *          - Bit to check in ParamPresenceMask - <B>0x01</B>
 * 
 *  @For_5G
 */
typedef struct {
    uint8_t default_configured_nssai_length;
    char *pDefault_configured_nssai;
    swi_uint256_t ParamPresenceMask;
} unpack_swinas_GetDefaultNssai_t;

/**
 * \ingroup nas
 * 
 * pack function to get Default NSSAI Info
 * @param[in,out] pCtx qmi request context
 * @param[out] pReqBuf qmi request buffer
 * @param[out] pLen qmi request length
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 * 
 * @For_5G
 */

int pack_swinas_GetDefaultNssai(
    pack_qmi_t * pCtx,
    uint8_t * pReqBuf,
    uint16_t * pLen);

/**
 * \ingroup nas
 * 
 * unpack function to get Default NSSAI info
 * 
 * @param[in] pResp qmi response
 * @param[in] respLen qmi response length
 * @param[out] pOutput unpacked response
 *
 * @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 * @sa     See qmerrno.h for eQCWWAN_xxx error values
 * 
 * @For_5G
 */
int unpack_swinas_GetDefaultNssai(
    uint8_t * pResp,
    uint16_t respLen,
    unpack_swinas_GetDefaultNssai_t *pOutput);

typedef unpack_swinas_GetDefaultNssai_t pack_swinas_SetDefaultNssai_t;

/**
 * \ingroup nas
 * 
 *  Pack function for Set Default NSSAI.
 * 
 *  @param[in,out] pCtx qmi request context
 *  @param[out] pReqBuf qmi request buffer
 *  @param[out] pLen qmi request length
 *  @param[in] pReqParam request prarmeters
 *
 *  @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 *  @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 *  @note   Device Supported: MC75xx, EM75xx and WP76xx.
 *
 *  @For_5G
 *
 */
int pack_swinas_SetDefaultNssai(
	pack_qmi_t * pCtx,
	uint8_t * pReqBuf,
	uint16_t * pLen,
	pack_swinas_SetDefaultNssai_t *pReqParam);

typedef unpack_result_t  unpack_swinas_SetDefaultNssai_t;

/**
 * \ingroup nas
 * 
 *  Unpack function for Set PDP Profile Settings Ext.
 * 
 *  @param[in]   pResp       qmi response from modem
 *  @param[in]   respLen     qmi response length
 *  @param[out]  pOutput     unpacked response
 *
 *  @return eQCWWAN_ERR_NONE on success, eQCWWAN_xxx error value otherwise
 *
 *  @sa     See qmerrno.h for eQCWWAN_xxx error values
 *
 *  @For_5G
 *
 */
int unpack_swinas_SetDefaultNssai(
	uint8_t * pResp,
	uint16_t respLen,
	unpack_swinas_SetDefaultNssai_t *pOutput);

#ifdef __cplusplus
} /* extern "C" { */
#endif

#endif
